<?php
// JEDI Security • Matrix Uploader (v2)
// - AJAX upload + progress
// - On success reloads with ?link=... and auto-copies the link

$link = $_GET['link'] ?? '';
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>JEDI Security • File Share</title>
  <meta name="robots" content="noindex,nofollow" />

  <style>
    :root{
      --bg:#000;
      --matrix:#00ff66;
      --ice:#7fd7ff;
      --muted:rgba(0,255,102,.65);
      --panel:rgba(0,0,0,.62);
      --border:rgba(0,255,102,.35);
      --shadow:0 14px 45px rgba(0,0,0,.55);
      --radius:18px;
      --mono: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace;
    }
    *{box-sizing:border-box}
    html,body{height:100%}
    body{
      margin:0;
      background:var(--bg);
      color:var(--matrix);
      font-family:var(--mono);
      overflow-x:hidden;
    }

    #matrix{
      position:fixed;
      inset:0;
      z-index:0;
      opacity:.22;
      filter:saturate(1.1) contrast(1.05);
    }

    .wrap{
      position:relative;
      z-index:2;
      min-height:100%;
      display:flex;
      align-items:center;
      justify-content:center;
      padding:28px 14px;
    }

    .card{
      width:min(880px, 100%);
      background:var(--panel);
      border:1px solid var(--border);
      border-radius:var(--radius);
      box-shadow:var(--shadow);
      backdrop-filter: blur(10px);
      -webkit-backdrop-filter: blur(10px);
      padding:22px;
    }

    .top{
      display:flex;
      gap:14px;
      align-items:center;
      justify-content:space-between;
      flex-wrap:wrap;
      padding-bottom:10px;
      border-bottom:1px dashed rgba(0,255,102,.25);
      margin-bottom:16px;
    }

    .brand{
      display:flex;
      align-items:center;
      gap:12px;
    }

    .logo{
      width:44px;height:44px;
      border-radius:12px;
      border:1px solid rgba(127,215,255,.35);
      background:
        radial-gradient(circle at 30% 30%, rgba(127,215,255,.55), transparent 55%),
        radial-gradient(circle at 70% 70%, rgba(0,255,102,.35), transparent 55%),
        linear-gradient(135deg, rgba(0,255,102,.16), rgba(127,215,255,.10));
      display:grid;
      place-items:center;
      box-shadow:0 10px 30px rgba(0,0,0,.5);
    }
    .logo span{
      font-weight:900;
      letter-spacing:.5px;
      color:var(--ice);
      text-shadow: 0 0 16px rgba(127,215,255,.55);
    }

    h1{
      margin:0;
      font-size:18px;
      letter-spacing:1px;
      color:var(--ice);
      text-shadow: 0 0 18px rgba(127,215,255,.35);
    }
    .sub{
      margin:2px 0 0 0;
      font-size:12px;
      color:var(--muted);
    }

    .pill{
      font-size:12px;
      color:rgba(127,215,255,.9);
      border:1px solid rgba(127,215,255,.28);
      border-radius:999px;
      padding:8px 12px;
      background:rgba(127,215,255,.06);
    }

    .grid{display:grid;grid-template-columns:1fr;gap:14px}
    .panel{
      border:1px solid rgba(0,255,102,.22);
      border-radius:16px;
      padding:16px;
      background:rgba(0,0,0,.42);
    }
    .row{display:flex;gap:10px;align-items:center;flex-wrap:wrap}
    .file{
      flex:1;
      min-width:220px;
      border:1px solid rgba(0,255,102,.25);
      border-radius:12px;
      padding:10px 12px;
      background:rgba(0,0,0,.55);
      color:var(--ice);
      outline:none;
    }
    .btn{
      border:1px solid rgba(127,215,255,.38);
      color:var(--ice);
      background:linear-gradient(135deg, rgba(127,215,255,.10), rgba(0,255,102,.08));
      padding:10px 14px;
      border-radius:12px;
      cursor:pointer;
      font-weight:700;
      letter-spacing:.3px;
      transition: transform .08s ease, border-color .12s ease, background .12s ease;
      text-decoration:none;
      display:inline-flex;
      align-items:center;
      gap:8px;
    }
    .btn:hover{ transform: translateY(-1px); border-color:rgba(127,215,255,.55); }
    .btn:active{ transform: translateY(0px); }
    .btn:disabled{opacity:.5;cursor:not-allowed;transform:none}

    .meta{font-size:12px;color:var(--muted);margin-top:10px;line-height:1.5}
    .progressWrap{
      margin-top:12px;
      border:1px solid rgba(0,255,102,.22);
      border-radius:999px;
      overflow:hidden;
      height:14px;
      background:rgba(0,0,0,.55);
    }
    .bar{
      height:100%;
      width:0%;
      background:linear-gradient(90deg, rgba(0,255,102,.0), rgba(0,255,102,.85), rgba(127,215,255,.85));
      transition: width .12s ease;
    }

    .linkBox{display:flex;gap:10px;align-items:center;flex-wrap:wrap;margin-top:10px}
    .link{
      flex:1;
      min-width:240px;
      border:1px solid rgba(127,215,255,.35);
      border-radius:12px;
      padding:10px 12px;
      background:rgba(0,0,0,.55);
      color:var(--ice);
      outline:none;
    }

    .toast{
      position:fixed;
      right:14px;
      bottom:14px;
      z-index:3;
      padding:12px 14px;
      border-radius:14px;
      border:1px solid rgba(127,215,255,.35);
      background:rgba(0,0,0,.72);
      color:var(--ice);
      box-shadow:0 16px 55px rgba(0,0,0,.6);
      display:none;
      max-width:min(520px, calc(100vw - 28px));
    }
    .toast strong{ color:var(--matrix); }
    .small{ font-size:12px; color:rgba(127,215,255,.8); margin-top:6px; }

    .footer{
      margin-top:14px;
      font-size:11px;
      color:rgba(0,255,102,.55);
      display:flex;
      justify-content:space-between;
      gap:10px;
      flex-wrap:wrap;
    }
    a{ color:var(--ice); text-decoration:none; border-bottom:1px dotted rgba(127,215,255,.35); }
    a:hover{ border-bottom-color:rgba(127,215,255,.7); }
    .warn{ color:rgba(255,120,120,.9); border-color:rgba(255,120,120,.35); background:rgba(255,120,120,.06); }
  </style>
</head>
<body>
  <canvas id="matrix"></canvas>

  <div class="wrap">
    <div class="card">
      <div class="top">
        <div class="brand">
          <div class="logo" aria-hidden="true"><span>JS</span></div>
          <div>
            <h1>JEDI Security • File Share</h1>
            <div class="sub">Upload → share link → auto-copy on reload. No accounts. No friction.</div>
          </div>
        </div>
        <div class="pill">Host: <span style="color:var(--matrix)">jedisecurity.cloud</span></div>
      </div>

      <div class="grid">
        <div class="panel">
          <form id="uform" action="upload.php" method="post" enctype="multipart/form-data">
            <div class="row">
              <input class="file" id="file" type="file" name="file" required />
              <button class="btn" id="uploadBtn" type="submit">UPLOAD</button>
              <button class="btn" id="resetBtn" type="button" title="Clear">RESET</button>
              <a class="btn" href="browse.php" title="Browse uploads">BROWSE</a>
            </div>

            <div class="progressWrap" aria-label="Upload progress">
              <div class="bar" id="bar"></div>
            </div>

            <div class="meta">
              <div>⚙️ Indexed: every upload appends metadata to <strong>/uploads/index.json</strong>.</div>
              <div>🧊 Theme: Matrix green with ice-blue accents.</div>
            </div>
          </form>
        </div>

        <div class="panel" id="resultPanel" style="<?php echo $link ? '' : 'display:none;'; ?>">
          <div style="color:var(--ice); font-weight:800; letter-spacing:.6px;">SHARE LINK</div>
          <div class="linkBox">
            <input class="link" id="shareLink" value="<?php echo htmlspecialchars($link); ?>" readonly />
            <button class="btn" id="copyBtn" type="button">COPY</button>
            <a class="btn" id="openBtn" href="<?php echo htmlspecialchars($link); ?>" target="_blank" rel="noopener">OPEN</a>
            <a class="btn" id="dlBtn" href="<?php echo htmlspecialchars($link); ?>" target="_blank" rel="noopener">DOWNLOAD</a>
          </div>
          <div class="meta">Auto-copy triggers on page reload when a new link is present.</div>
        </div>

        <div class="footer">
          <div>© <?php echo date('Y'); ?> JEDI Security</div>
          <div><a href="README.txt">Setup</a> • <a href="browse.php">Uploads Browser</a></div>
        </div>
      </div>
    </div>
  </div>

  <div class="toast" id="toast"></div>

<script>
/* Matrix rain — lightweight */
(function(){
  const c = document.getElementById('matrix');
  const ctx = c.getContext('2d');
  const chars = 'アイウエオカキクケコサシスセソタチツテトナニヌネノハヒフヘホマミムメモヤユヨラリルレロワヲン0123456789JEDISEC';
  let w,h,cols,drops,fontSize=16;

  function resize(){
    w = c.width = window.innerWidth;
    h = c.height = window.innerHeight;
    cols = Math.floor(w / fontSize);
    drops = Array(cols).fill(1);
  }
  window.addEventListener('resize', resize, {passive:true});
  resize();

  function draw(){
    ctx.fillStyle = 'rgba(0,0,0,0.06)';
    ctx.fillRect(0,0,w,h);

    ctx.fillStyle = 'rgba(0,255,102,0.85)';
    ctx.font = fontSize + 'px monospace';

    for(let i=0;i<drops.length;i++){
      const text = chars[Math.floor(Math.random()*chars.length)];
      ctx.fillText(text, i*fontSize, drops[i]*fontSize);
      if(drops[i]*fontSize > h && Math.random() > 0.975) drops[i]=0;
      drops[i]++;
    }
    requestAnimationFrame(draw);
  }
  requestAnimationFrame(draw);
})();

/* Upload + reload with link + auto-copy */
const form = document.getElementById('uform');
const fileInput = document.getElementById('file');
const uploadBtn = document.getElementById('uploadBtn');
const resetBtn = document.getElementById('resetBtn');
const bar = document.getElementById('bar');
const resultPanel = document.getElementById('resultPanel');
const shareLink = document.getElementById('shareLink');
const copyBtn = document.getElementById('copyBtn');
const openBtn = document.getElementById('openBtn');
const dlBtn = document.getElementById('dlBtn');
const toast = document.getElementById('toast');

function showToast(html, isWarn=false){
  toast.className = 'toast' + (isWarn ? ' warn' : '');
  toast.innerHTML = html;
  toast.style.display = 'block';
  clearTimeout(window.__toastTimer);
  window.__toastTimer = setTimeout(()=>toast.style.display='none', 3400);
}

async function copyText(text){
  try{
    await navigator.clipboard.writeText(text);
    return true;
  }catch(e){
    const tmp = document.createElement('textarea');
    tmp.value = text;
    document.body.appendChild(tmp);
    tmp.select();
    const ok = document.execCommand('copy');
    document.body.removeChild(tmp);
    return ok;
  }
}

resetBtn.addEventListener('click', ()=>{
  fileInput.value = '';
  bar.style.width = '0%';
  const url = new URL(window.location.href);
  url.searchParams.delete('link');
  window.history.replaceState({}, '', url.toString());
  resultPanel.style.display='none';
  showToast('<strong>Reset.</strong><div class="small">Ready for the next drop.</div>');
});

copyBtn?.addEventListener('click', async ()=>{
  const txt = shareLink.value;
  const ok = await copyText(txt);
  showToast(ok ? '<strong>Copied.</strong><div class="small">Link is in your clipboard.</div>'
               : '<strong>Copy failed.</strong><div class="small">Select the link and copy manually.</div>', !ok);
});

form.addEventListener('submit', (e)=>{
  e.preventDefault();
  if(!fileInput.files?.length){
    showToast('<strong>No file selected.</strong><div class="small">Pick a file first.</div>', true);
    return;
  }

  uploadBtn.disabled = true;
  bar.style.width = '0%';

  const xhr = new XMLHttpRequest();
  const data = new FormData();
  data.append('file', fileInput.files[0]);

  xhr.open('POST', 'upload.php', true);
  xhr.upload.addEventListener('progress', (evt)=>{
    if(evt.lengthComputable){
      const p = Math.max(1, Math.floor((evt.loaded/evt.total)*100));
      bar.style.width = p + '%';
    }
  });

  xhr.onreadystatechange = function(){
    if(xhr.readyState === 4){
      uploadBtn.disabled = false;

      let res = null;
      try{ res = JSON.parse(xhr.responseText); }catch(_){}
      if(xhr.status !== 200 || !res || !res.ok){
        const msg = (res && res.error) ? res.error : ('Upload failed (HTTP '+xhr.status+').');
        showToast('<strong>Upload blocked.</strong><div class="small">'+escapeHtml(msg)+'</div>', true);
        return;
      }

      // Reload with link so auto-copy happens on reload (per spec)
      window.location.href = 'index.php?link=' + encodeURIComponent(res.link);
    }
  };

  xhr.send(data);
});

function escapeHtml(str){
  return String(str).replace(/[&<>"']/g, s => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[s]));
}

// Auto-copy on reload if link exists
(function(){
  const url = new URL(window.location.href);
  const l = url.searchParams.get('link');
  if(!l) return;

  resultPanel.style.display = 'block';
  shareLink.value = l;
  openBtn.href = l;

  // Force "download" button to use download.php for attachment headers
  const dlUrl = 'download.php?path=' + encodeURIComponent(l.split('/uploads/')[1] || '');
  dlBtn.href = dlUrl;

  copyText(l).then(ok=>{
    showToast(
      ok
        ? '<strong>Copied.</strong><div class="small">Share link auto-copied on reload.</div>'
        : '<strong>Copy blocked.</strong><div class="small">Tap COPY to retry.</div>',
      !ok
    );
  });
})();
</script>

</body>
</html>
