<?php
// JEDI Security • Upload handler (v2)
// - Returns JSON (AJAX)
// - Stores files under /uploads/YYYY/MM/
// - Appends metadata to /uploads/index.json (flat-file DB with flock)
// - Blocks risky extensions

header('Content-Type: application/json; charset=utf-8');

function respond($ok, $payload = []) {
  echo json_encode(array_merge(['ok'=>$ok], $payload), JSON_UNESCAPED_SLASHES);
  exit;
}

if (!isset($_FILES['file'])) respond(false, ['error'=>'No file received.']);

$maxBytes = 1024 * 1024 * 1024 * 2; // 2 GB default
$f = $_FILES['file'];

if ($f['error'] !== UPLOAD_ERR_OK) respond(false, ['error'=>'Upload error code: '.$f['error']]);
if ($f['size'] <= 0) respond(false, ['error'=>'Empty file.']);
if ($f['size'] > $maxBytes) respond(false, ['error'=>'File too large. Max: '.number_format($maxBytes).' bytes.']);

$origName = $f['name'] ?? 'file';
$origName = preg_replace('/[^\w\.\- ]+/u', '_', $origName);
$ext = strtolower(pathinfo($origName, PATHINFO_EXTENSION));

$blocked = [
  'php','phtml','php3','php4','php5','phar','cgi','pl','py','rb','sh','bash','zsh',
  'js','jsx','ts','tsx','html','htm','shtml','xhtml','hta','asp','aspx','jsp','jar',
  'exe','dll','bat','cmd','com','msi','vbs','ps1','scr','reg','apk'
];
if ($ext && in_array($ext, $blocked, true)) respond(false, ['error'=>'Blocked file type: .'.$ext]);

$finfo = finfo_open(FILEINFO_MIME_TYPE);
$mime = $finfo ? finfo_file($finfo, $f['tmp_name']) : '';
if ($finfo) finfo_close($finfo);

$year = date('Y');
$month = date('m');

$destDir = __DIR__ . '/uploads/' . $year . '/' . $month . '/';
if (!is_dir($destDir) && !mkdir($destDir, 0777, true)) respond(false, ['error'=>'Failed to create upload folder.']);

$rand = bin2hex(random_bytes(16));
$storedName = $ext ? ($rand.'.'.$ext) : $rand;
$destPath = $destDir . $storedName;

if (!move_uploaded_file($f['tmp_name'], $destPath)) respond(false, ['error'=>'Failed to move uploaded file.']);

$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
$host = $_SERVER['HTTP_HOST'] ?? 'localhost';

// store web path relative to site root
$webPath = 'uploads/' . $year . '/' . $month . '/' . $storedName;
$link = $scheme . $host . '/' . $webPath;

// Append to JSON index (newline-delimited JSON would be faster at scale, but you asked JSON index)
$indexFile = __DIR__ . '/uploads/index.json';
$entry = [
  'id' => bin2hex(random_bytes(8)),
  'orig' => $origName,
  'path' => $webPath,             // e.g. uploads/2026/01/abc.png
  'bytes' => (int)$f['size'],
  'mime' => $mime,
  'ts' => time()
];

$fh = @fopen($indexFile, 'c+');
if ($fh) {
  if (flock($fh, LOCK_EX)) {
    $raw = stream_get_contents($fh);
    $data = [];
    if ($raw !== false && trim($raw) !== '') {
      $decoded = json_decode($raw, true);
      if (is_array($decoded)) $data = $decoded;
    }
    $data[] = $entry;
    ftruncate($fh, 0);
    rewind($fh);
    fwrite($fh, json_encode($data, JSON_UNESCAPED_SLASHES));
    fflush($fh);
    flock($fh, LOCK_UN);
  }
  fclose($fh);
}

respond(true, [
  'link' => $link,
  'name' => $origName,
  'mime' => $mime,
  'bytes' => (int)$f['size']
]);
