<?php
// JEDI Security • Uploads Browser (v2)
// - Reads /uploads/index.json
// - Search + sort + pagination
// - Select rows -> Copy selected links / Download selected (ZIP)
// - Copy ALL links (current filtered result set)

$indexFile = __DIR__ . '/uploads/index.json';
$uploadsFs = __DIR__ . '/';
$uploadsWebBase = ''; // paths stored include "uploads/...." already

$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
$host = $_SERVER['HTTP_HOST'] ?? 'localhost';
$baseUrl = $scheme . $host . '/';

$q = trim($_GET['q'] ?? '');
$sort = $_GET['sort'] ?? 'ts';         // ts | orig | bytes | path
$order = $_GET['order'] ?? 'desc';     // asc | desc
$page = max(1, (int)($_GET['page'] ?? 1));
$per = (int)($_GET['per'] ?? 50);
if (!in_array($per, [25,50,100,200], true)) $per = 50;

$data = [];
if (is_file($indexFile)) {
  $raw = file_get_contents($indexFile);
  $decoded = json_decode($raw, true);
  if (is_array($decoded)) $data = $decoded;
}

// Filter
$filtered = [];
$ql = strtolower($q);
foreach ($data as $row) {
  if (!is_array($row)) continue;
  $path = $row['path'] ?? '';
  $orig = $row['orig'] ?? '';
  if (!is_string($path) || $path === '') continue;

  if ($q !== '') {
    $hay = strtolower($orig . ' ' . $path);
    if (strpos($hay, $ql) === false) continue;
  }

  // derive rel path under /uploads for download endpoints: "YYYY/MM/file.ext"
  $rel = $path;
  if (str_starts_with($rel, 'uploads/')) $rel = substr($rel, 8);

  $row['_rel'] = $rel;
  $row['_url'] = $baseUrl . $path;
  $filtered[] = $row;
}

// Sort
$validSort = ['ts','orig','bytes','path'];
if (!in_array($sort, $validSort, true)) $sort = 'ts';
usort($filtered, function($a,$b) use ($sort, $order){
  $va = $a[$sort] ?? '';
  $vb = $b[$sort] ?? '';
  if ($sort === 'bytes' || $sort === 'ts') { $va = (int)$va; $vb = (int)$vb; }
  $cmp = $va <=> $vb;
  return ($order === 'asc') ? $cmp : -$cmp;
});

$total = count($filtered);
$pages = max(1, (int)ceil($total / $per));
if ($page > $pages) $page = $pages;

$start = ($page - 1) * $per;
$rows = array_slice($filtered, $start, $per);

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function bytes_fmt($n){
  $n = (int)$n;
  $u=['B','KB','MB','GB','TB']; $i=0;
  while($n>=1024 && $i<count($u)-1){ $n/=1024; $i++; }
  return round($n, $i?2:0).' '.$u[$i];
}

function qs($overrides=[]){
  $params = $_GET;
  foreach($overrides as $k=>$v){
    if ($v === null) unset($params[$k]);
    else $params[$k] = $v;
  }
  return '?' . http_build_query($params);
}
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>JEDI Security • Uploads Browser</title>
<meta name="robots" content="noindex,nofollow" />
<style>
  :root{
    --bg:#000; --matrix:#00ff66; --ice:#7fd7ff;
    --panel:rgba(0,0,0,.62); --border:rgba(0,255,102,.28);
    --mono: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace;
  }
  *{box-sizing:border-box}
  body{margin:0;background:var(--bg);color:var(--matrix);font-family:var(--mono)}
  .wrap{max-width:1250px;margin:0 auto;padding:18px}
  .card{background:var(--panel);border:1px solid var(--border);border-radius:18px;padding:16px}
  h1{margin:0 0 10px 0;color:var(--ice);font-size:18px;letter-spacing:.8px}
  .sub{color:rgba(0,255,102,.65);font-size:12px;margin-bottom:12px;display:flex;gap:10px;flex-wrap:wrap;align-items:center;justify-content:space-between}
  form.filters{display:flex;gap:10px;flex-wrap:wrap;margin:10px 0 10px}
  input,select,button,a.btn{
    padding:10px 12px;border-radius:12px;background:rgba(0,0,0,.55);
    border:1px solid rgba(0,255,102,.25);color:var(--ice);text-decoration:none
  }
  button{cursor:pointer;border-color:rgba(127,215,255,.35)}
  a.btn{display:inline-flex;align-items:center;gap:8px;border-color:rgba(127,215,255,.35)}
  table{width:100%;border-collapse:collapse;margin-top:10px}
  th,td{padding:10px;border-bottom:1px dashed rgba(0,255,102,.18);font-size:13px;vertical-align:top}
  th{color:rgba(127,215,255,.9);text-align:left}
  .muted{color:rgba(0,255,102,.6)}
  .pill{display:inline-block;padding:6px 10px;border-radius:999px;border:1px solid rgba(127,215,255,.25);background:rgba(127,215,255,.06);color:rgba(127,215,255,.9);font-size:12px}
  .actions{white-space:nowrap}
  .pager{display:flex;gap:10px;flex-wrap:wrap;align-items:center;justify-content:space-between;margin-top:14px}
  .rowActions{display:flex;gap:8px;flex-wrap:wrap;align-items:center;margin-top:10px}
  .small{font-size:12px;color:rgba(127,215,255,.85)}
</style>
</head>
<body>
<div class="wrap">
  <div class="card">
    <div class="sub">
      <div>
        <h1>JEDI Security • Uploads Browser</h1>
        <div class="muted">JSON index + pagination + batch copy + batch download</div>
      </div>
      <div class="pill"><?= $total ?> file(s)</div>
    </div>

    <form class="filters" method="get">
      <input name="q" placeholder="search name/path..." value="<?= h($q) ?>" />
      <select name="sort">
        <option value="ts" <?= $sort==='ts'?'selected':'' ?>>sort: date</option>
        <option value="orig" <?= $sort==='orig'?'selected':'' ?>>sort: original name</option>
        <option value="bytes" <?= $sort==='bytes'?'selected':'' ?>>sort: size</option>
        <option value="path" <?= $sort==='path'?'selected':'' ?>>sort: path</option>
      </select>
      <select name="order">
        <option value="desc" <?= $order==='desc'?'selected':'' ?>>desc</option>
        <option value="asc" <?= $order==='asc'?'selected':'' ?>>asc</option>
      </select>
      <select name="per">
        <option value="25" <?= $per===25?'selected':'' ?>>25/page</option>
        <option value="50" <?= $per===50?'selected':'' ?>>50/page</option>
        <option value="100" <?= $per===100?'selected':'' ?>>100/page</option>
        <option value="200" <?= $per===200?'selected':'' ?>>200/page</option>
      </select>
      <button type="submit">Apply</button>
      <a class="btn" href="browse.php">Reset</a>
      <a class="btn" href="index.php">Uploader</a>
    </form>

    <div class="rowActions">
      <button type="button" onclick="toggleAll(true)">Select all (page)</button>
      <button type="button" onclick="toggleAll(false)">Clear (page)</button>
      <button type="button" onclick="copySelected()">Copy selected links</button>
      <button type="button" onclick="copyAllFiltered()">Copy ALL links (filtered)</button>
      <button type="button" onclick="downloadSelected()">Download selected (ZIP)</button>
      <span class="small" id="selCount">0 selected</span>
    </div>

    <table>
      <thead>
        <tr>
          <th style="width:34px;"></th>
          <th>Original</th>
          <th>Stored path</th>
          <th>Size</th>
          <th>Uploaded</th>
          <th>Actions</th>
        </tr>
      </thead>
      <tbody>
      <?php foreach($rows as $r): ?>
        <tr>
          <td><input type="checkbox" class="rowchk" data-url="<?= h($r['_url']) ?>" data-rel="<?= h($r['_rel']) ?>" onchange="updateSel()" /></td>
          <td><?= h($r['orig'] ?? '') ?></td>
          <td class="muted"><?= h($r['path'] ?? '') ?></td>
          <td><?= h(bytes_fmt($r['bytes'] ?? 0)) ?></td>
          <td><?= isset($r['ts']) ? date('Y-m-d H:i', (int)$r['ts']) : '' ?></td>
          <td class="actions">
            <a class="btn" href="<?= h($r['_url']) ?>" target="_blank" rel="noopener">Open</a>
            <a class="btn" href="download.php?path=<?= h(rawurlencode($r['_rel'])) ?>">Download</a>
            <button type="button" onclick="copyOne('<?= h($r['_url']) ?>')">Copy</button>
          </td>
        </tr>
      <?php endforeach; ?>
      </tbody>
    </table>

    <div class="pager">
      <div class="small">Page <?= $page ?> / <?= $pages ?> • Showing <?= count($rows) ?> of <?= $total ?></div>
      <div style="display:flex;gap:10px;flex-wrap:wrap;align-items:center;">
        <a class="btn" href="<?= h('browse.php'.qs(['page'=>1])) ?>">« First</a>
        <a class="btn" href="<?= h('browse.php'.qs(['page'=>max(1,$page-1)])) ?>">‹ Prev</a>
        <a class="btn" href="<?= h('browse.php'.qs(['page'=>min($pages,$page+1)])) ?>">Next ›</a>
        <a class="btn" href="<?= h('browse.php'.qs(['page'=>$pages])) ?>">Last »</a>
      </div>
    </div>

  </div>
</div>

<script>
function rowChecks(){ return Array.from(document.querySelectorAll('.rowchk')); }
function selected(){ return rowChecks().filter(x=>x.checked); }
function updateSel(){
  const n = selected().length;
  document.getElementById('selCount').textContent = n + ' selected';
}
function toggleAll(on){
  rowChecks().forEach(x=>x.checked = on);
  updateSel();
}

async function copyText(text){
  try{ await navigator.clipboard.writeText(text); return true; }
  catch(e){
    const ta=document.createElement('textarea');
    ta.value=text; document.body.appendChild(ta); ta.select();
    const ok=document.execCommand('copy'); document.body.removeChild(ta);
    return ok;
  }
}

function copyOne(url){
  copyText(url).then(()=>alert('Copied ✅'));
}

function copySelected(){
  const urls = selected().map(x=>x.dataset.url);
  if(!urls.length){ alert('No files selected.'); return; }
  copyText(urls.join('\n')).then(()=>alert('Copied '+urls.length+' link(s) ✅'));
}

function copyAllFiltered(){
  // Server embeds the filtered URL list via JSON in a script tag below
  const urls = window.__ALL_FILTERED_URLS || [];
  if(!urls.length){ alert('No results.'); return; }
  copyText(urls.join('\n')).then(()=>alert('Copied ALL '+urls.length+' link(s) ✅'));
}

function downloadSelected(){
  const rels = selected().map(x=>x.dataset.rel);
  if(!rels.length){ alert('No files selected.'); return; }

  const form = document.createElement('form');
  form.method = 'POST';
  form.action = 'bulk_download.php';

  const inp = document.createElement('input');
  inp.type = 'hidden';
  inp.name = 'files';
  inp.value = JSON.stringify(rels);

  form.appendChild(inp);
  document.body.appendChild(form);
  form.submit();
  form.remove();
}

// Embed ALL filtered URLs for "copy all links"
window.__ALL_FILTERED_URLS = <?= json_encode(array_map(fn($r)=>$r['_url'], $filtered), JSON_UNESCAPED_SLASHES) ?>;

updateSel();
</script>
</body>
</html>
